# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

# pylint: disable=pointless-string-statement

import os
import logging

import sqlalchemy as sa
from sqlalchemy.exc import OperationalError
from alembic import op

from lvestats.orm.const import LVE_STATS_2_TABLENAME_PREFIX, SERVER_ID_LENGTH

"""snapshot changes

Revision ID: 54d3ae3043dd
Revises: 15fed017ca0b
Create Date: 2016-08-17 06:13:24.019719

"""

# revision identifiers, used by Alembic.
revision = '54d3ae3043dd'
down_revision = '15fed017ca0b'
branch_labels = None
depends_on = None

INCIDENT_TABLE = LVE_STATS_2_TABLENAME_PREFIX + 'incident'
SNAPSHOT_TABLE = LVE_STATS_2_TABLENAME_PREFIX + 'snapshot'

logger = logging.getLogger(f'alembic.script.{os.path.basename(__name__)}')


def upgrade():
    # Lets clean up old data, we will not migrate it on upgrade or downgrade
    op.execute("delete from " + INCIDENT_TABLE)

    op.add_column(INCIDENT_TABLE, sa.Column('dump_time', sa.Integer(), nullable=True))
    op.add_column(INCIDENT_TABLE, sa.Column('snapshot_count', sa.Integer(), nullable=True))
    op.drop_table(SNAPSHOT_TABLE)


def downgrade():
    # due to sqlite not supporting drop column... lets re-create it
    # we wanted to remove data anyway

    op.drop_table(INCIDENT_TABLE)
    op.create_table(INCIDENT_TABLE,
                    sa.Column('uid', sa.INTEGER(), index=True, primary_key=True, autoincrement=False),
                    sa.Column('server_id', sa.VARCHAR(length=SERVER_ID_LENGTH), index=True, primary_key=True),
                    sa.Column('incident_start_time', sa.INTEGER(), index=True, primary_key=True, autoincrement=False),
                    sa.Column('incident_end_time', sa.INTEGER(), index=True)
                    )
    try:
        op.create_table(SNAPSHOT_TABLE,
                        sa.Column('uid', sa.INTEGER(), nullable=False),
                        sa.Column('server_id', sa.VARCHAR(length=255), nullable=False),
                        sa.Column('incident_start_time', sa.INTEGER(), nullable=False),
                        sa.Column('dump_time', sa.INTEGER(), nullable=False),
                        sa.Column('snap_proc', sa.TEXT(), nullable=True),
                        sa.Column('snap_sql', sa.TEXT(), nullable=True),
                        sa.Column('snap_http', sa.TEXT(), nullable=True),
                        sa.Column('snap_faults', sa.TEXT(), nullable=True),
                        sa.PrimaryKeyConstraint('uid', 'server_id', 'incident_start_time', 'dump_time')
                        )
    except OperationalError as ex:
        # for backward compatibility with old lve-create-db (lve-stats <= 2.3-x)
        if f'table {SNAPSHOT_TABLE} already exists' in ex.message:
            logger.warning(str(ex))
        else:
            raise
