# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import os

CONFIG_PATH_v1 = '/etc/sysconfig/lvestats'
CONFIG_PATH_v2 = '/etc/sysconfig/lvestats2'

NOTIFY_PATH_v1 = '/etc/sysconfig/cloudlinux-notify'
NOTIFY_PATH_v2 = '/etc/sysconfig/lvestats.config/StatsNotifier.cfg'

IMPORT_STATUS_FILE = '/usr/share/lve-stats/utils/import-settings-status'

V1_DB_MIIGRATOR_CONFIG = '/etc/sysconfig/lvestats.config/V1DBMigrator.cfg'
V1_SQLITE_DB = 'sqlite:////var/lve/lveinfo.db'
V1_DB_MIIGRATOR_PLUGIN_SRC = '/usr/share/lve-stats/plugins.other/v1_db_migrator.py'
V1_DB_MIIGRATOR_PLUGIN_LINK = '/usr/share/lve-stats/plugins/v1_db_migrator.py'
V1_KEY = 'v1_connect_string'


def init_v1_db_migrator(config):
    if V1_KEY not in config:
        print('Cannot find db info to enable migration plugin')
        return

    if 'server_id' not in config:
        config['server_id'] = 'localhost'

    with open(V1_DB_MIIGRATOR_CONFIG, 'w', encoding='utf-8') as cfg:
        cfg.write(f'v1_server_id={config["server_id"]}\nv1_connect_string={config[V1_KEY]}\n')

    try:
        if not os.path.lexists(V1_DB_MIIGRATOR_PLUGIN_LINK):
            os.symlink(V1_DB_MIIGRATOR_PLUGIN_SRC, V1_DB_MIIGRATOR_PLUGIN_LINK)
    except (IOError, OSError) as e:
        print(f"Unable to enable db migration plugin: {e}, continue: ")


def get_import_status():
    if os.path.exists(IMPORT_STATUS_FILE):
        with open(IMPORT_STATUS_FILE, 'r', encoding='utf-8') as f:
            v = f.read()
        try:
            return int(v)
        except ValueError as e:
            raise RuntimeError(f'Invalid status format for file {IMPORT_STATUS_FILE}') from e
    else:
        return 0


def set_import_status(flag):
    with open(IMPORT_STATUS_FILE, 'w', encoding='utf-8') as f:
        f.write(f'{flag}')


def get_config_path(path):
    if os.path.exists(path):
        return path
    elif os.path.exists(path + '.rpmsave'):
        return path + '.rpmsave'
    return ''


def get_connection_string(lvestats_config):
    if 'db_type' in lvestats_config:
        if 'connect_string' in lvestats_config and lvestats_config['db_type'] in ('mysql', 'postgresql'):
            connect_string = lvestats_config['connect_string'].split(':')
            host = connect_string[0]
            database = connect_string[1]
            user = connect_string[2]
            password = connect_string[3]
            lvestats_config['connect_string'] = f"{user}:{password}@{host}/{database}"
            lvestats_config[V1_KEY] = f"{lvestats_config['db_type']}://{lvestats_config['connect_string']}"
        elif lvestats_config['db_type'] == 'sqlite':
            lvestats_config[V1_KEY] = V1_SQLITE_DB
