"""General utilities for working with subprocesses."""
import signal
import subprocess
from subprocess import PIPE  # noqa: F401

__all__ = ["PIPE", "CalledProcessError", "check_output"]


class CalledProcessError(subprocess.CalledProcessError):
    """Add stdout,stderr to str representation"""

    def __str__(self):
        if self.returncode and self.returncode < 0:
            try:
                return "Command '%s' died with %r.\nStdout: %s\nStderr: %s" % (
                    self.cmd,
                    signal.Signals(
                        -self.returncode
                    ),  # noqa E501 pylint: disable=E1101
                    self.stdout,
                    self.stderr,
                )
            except ValueError:
                return (
                    "Command '%s' died with unknown signal %d."
                    "\nStdout: %s\nStderr: %s"
                    % (self.cmd, -self.returncode, self.stdout, self.stderr)
                )
        else:
            return (
                "Command '%s' returned non-zero exit status %d."
                "\nStdout: %s\nStderr: %s"
                % (self.cmd, self.returncode, self.stdout, self.stderr)
            )


def check_output(*args, **kwargs):
    """A wrapper for stdlib subprocess.check_output.

    Include stdout/stderr in error message.
    """
    try:
        return subprocess.check_output(*args, **kwargs)
    except subprocess.CalledProcessError as e:
        raise CalledProcessError(
            e.returncode, e.cmd, e.stdout, e.stderr
        ) from None
