"""High-level support for working with threads in asyncio

Modified from Python 3.10 stdlib
https://github.com/python/cpython/blob/b11a951f16f0603d98de24fee5c023df83ea552c/Lib/asyncio/threads.py
(the license GPL-compatible but doesn't require to open-source either).
"""

import functools
import contextvars

from asyncio import events


__all__ = ("to_thread",)


async def to_thread(func, /, *args, **kwargs):
    """Asynchronously run function *func* in a separate thread.
    Any *args and **kwargs supplied for this function are directly passed
    to *func*. Also, the current :class:`contextvars.Context` is propogated,
    allowing context variables from the main thread to be accessed in the
    separate thread.
    Return a coroutine that can be awaited to get the eventual result of *func*
    """
    loop = events.get_running_loop()
    ctx = contextvars.copy_context()
    func_call = functools.partial(ctx.run, func, *args, **kwargs)
    return await loop.run_in_executor(None, func_call)
