"""Notify systemd about process state"""
import logging
import os

from defence360agent.contracts.config import ANTIVIRUS_MODE


logger = logging.getLogger(__name__)


class AgentState(object):
    """Allowed agent state for notifying systemd."""

    READY = "READY=1"
    STARTING = "STATUS=Starting main process"
    MIGRATING = "STATUS=Applying database migrations"
    DAEMONIZED = "STATUS=Demonized"


def notify(state):
    """
    Send notification to systemd, allowed formats described here
    https://www.freedesktop.org/software/systemd/man/sd_notify.html

    For example:

        notify("STATUS=Almost ready")

    """
    if ANTIVIRUS_MODE:
        return
    try:
        import sdnotify
    # We've made this import inside function deliberately because some
    # systems don't have systemd and stdnotify thus
    except ImportError as e:
        logger.info("Can't import sdnotify, reason: %s", e)
    else:
        if "NOTIFY_SOCKET" not in os.environ:
            logger.warning(
                "$NOTIFY_SOCKET environment variable is not set. It is set by"
                " service manager for supervised processes for status and"
                " start-up completion notification. Nowhere to send the"
                " status."
            )
        else:
            try:
                n = sdnotify.SystemdNotifier(debug=True)
                n.notify(state)

            except Exception as e:  # NOSONAR
                logger.exception(
                    "some problem has occurred during notifying of"
                    " systemd: %s",
                    e,
                )
