# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2025 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
import os
import subprocess
from typing import Optional


def svg_to_png(svg_data: str) -> bytes:
    """
    Converts SVG data to PNG format using available converters.

    Args:
        svg_data (str): The SVG data to convert.

    Returns:
        bytes: The converted PNG data.

    Raises:
        RuntimeError: If no suitable converter is found.
    """
    # alt-ImageMagick is preferred over the native ImageMagick and rsvg.
    alt_imagemagick = _get_alt_imagemagick_binary()
    if alt_imagemagick:
        return _convert_with_imagemagick(svg_data, alt_imagemagick)

    rsvg_convert = _get_rsvg_convert_binary()
    if rsvg_convert:
        return _convert_with_rsvg(svg_data, rsvg_convert)

    imagemagick = _get_imagemagick_binary()
    if imagemagick:
        return _convert_with_imagemagick(svg_data, imagemagick)

    raise RuntimeError("No suitable SVG to PNG converter found")


def _get_alt_imagemagick_binary() -> Optional[str]:
    """
    Returns the path to the alt-ImageMagick binary if it exists, otherwise None.
    """
    alt_imagemagick = "/opt/alt/alt-ImageMagick/usr/bin/magick"
    if os.path.exists(alt_imagemagick):
        return alt_imagemagick
    return None


def _get_imagemagick_binary() -> Optional[str]:
    """
    Returns the path to the ImageMagick binary if it exists, otherwise None.
    """
    native_imagemagick = "/usr/bin/convert"
    if os.path.exists(native_imagemagick):
        return native_imagemagick
    return None


def _get_rsvg_convert_binary() -> Optional[str]:
    """
    Returns the path to the rsvg-convert binary if it exists, otherwise None.
    """
    rsvg_convert = "/usr/bin/rsvg-convert"
    if os.path.exists(rsvg_convert):
        return rsvg_convert
    return None


def _convert_with_rsvg(svg_data: str, rsvg_convert: str) -> bytes:
    """
    Converts SVG data to PNG format using rsvg-convert.

    Args:
        svg_data (str): The SVG data to convert.
        rsvg_convert (str): The path to the rsvg-convert binary.

    Returns:
        bytes: The converted PNG data.
    """
    cwd = '/var/lve/tmp' if os.getuid() == 0 else None
    with subprocess.Popen(
        [rsvg_convert, "-f", "png", "-b", "rgba(255, 255, 255, 255)"],
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        cwd=cwd
    ) as proc:
        png_data, _ = proc.communicate(input=svg_data.encode())
    return png_data


def _convert_with_imagemagick(svg_data: str, imagemagick: str) -> bytes:
    """
    Converts SVG data to PNG format using ImageMagick.

    Args:
        svg_data (str): The SVG data to convert.
        imagemagick (str): The path to the ImageMagick binary.

    Returns:
        bytes: The converted PNG data.
    """
    cwd = '/var/lve/tmp' if os.getuid() == 0 else None
    with subprocess.Popen(
        [imagemagick, "-", "png:-"],
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        cwd=cwd
    ) as proc:
        png_data, _ = proc.communicate(input=svg_data.encode())
    return png_data
