# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
from clcommon.lib import cledition

from clconfigure import task
from clconfigure.cagefs import clean_cagefs_without_lve_environment, reconfigure_cagefs
from clconfigure.clpro import disable_autotracing, trigger_manage_clplus
from clconfigure.files import update_cl_edition_file, remove_cl_edition_file
from clconfigure.limits import (
    initialize_lvestats,
    enable_lve_services,
    disable_lve_services
)
from clconfigure.linksafe import initialize_linksafe
from clconfigure.packages import (
    install_package,
    set_package_state,
    STATE_REMOVED,
    erase_package,
)


@task(f"Initial provisioning during transition of {cledition.get_cl_edition_readable()} into CloudLinux OS Shared")
def preprovision(edition):
    """
    Triggered before jwt token change in rhn-client-tools
    but already after successful registration with new edition.

    Designed to safely terminate things that should not be
    running as part of current edition but avoid license checks.
    """
    # erase of these packages automatically turns off features
    # and cleanups system
    set_package_state(STATE_REMOVED, 'accelerate-wp')
    set_package_state(STATE_REMOVED, 'alt-php-xray')
    # autotracing is not supported on shared
    disable_autotracing()
    clean_cagefs_without_lve_environment()


@task("Converting environment to CloudLinux OS Shared")
def provision(edition):
    """
    Converts CL Solo into "normal" CloudLinux by enabling
    some services and installing some packages.
    """
    remove_cl_edition_file()
    erase_package('cloudlinux-solo-meta')
    erase_package('cloudlinux-admin-meta')
    update_cl_edition_file()

    if not cledition.is_container():
        initialize_linksafe()
        initialize_lvestats()
        enable_lve_services()
    else:
        disable_lve_services()

    # Reinstall lve-utils:
    # - to apply correct sysctl parameters, see clsupergid_process.py
    # - to fix 'check_symlinkowngid' function in cldiag
    install_package('lve-utils', reinstall=True)
    # Reinstall lvemanager:
    # - to update linksafe group on the wrappers files
    #   within '/usr/share/l.v.e-manager/utils', see 'post' scriptlet
    # - as a workaround for CLOS-2394
    install_package('lvemanager', reinstall=True)
    reconfigure_cagefs()
    trigger_manage_clplus()
