# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from clcommon.utils import run_command
from lvestats.core.plugin import LveStatsPlugin

LVEPS_COMMAND = '/usr/sbin/lveps'


class ResMEMCollector(LveStatsPlugin):
    order = 1500  # should run after other collectors, but before analyzers
    timeout = 15
    """
    The goal of this plugin is to substitute physical memory readings from /proc/lve/list
    with RES data from /proc/* for each process within the LVE
    The data is extracted from lveps -p command

    """
    @staticmethod
    def parse_mem_per_lve(lines):
        """
        parse the output of lveps -p -n -o id:10,mem:15
        lveps returns memory in megabytes. The data for LVE is taken from /proc/lve/list
        the data for processes is taken from /proc/PID/... and is what we need.
        :param: lines list of lines, as outputed by lveps
        :return: dictionary with LVE id as keys, and memory in bytes as values
        """
        result = {}
        lve_id = None
        mem = 0
        for i in range(1, len(lines)):
            val = lines[i].split()
            if len(val) == 2:
                if lve_id is not None:
                    result[lve_id] = mem * 256  # the mem comes in MB, we need to convert it into num of 4k pages
                mem = 0
                lve_id = int(val[0])
            elif len(val) == 1:
                mem += float(val[0])
        if lve_id is not None:
            result[lve_id] = mem * 256  # the mem comes in MB, we need to convert it into num of 4k pages
        return result

    @staticmethod
    def get_mem_per_lve():
        """
        Get amount of memory processes inside LVE use
        :return: dictionary of LVE_ID <-> MEM
        """
        lveps_output = run_command([LVEPS_COMMAND, '-p', '-n', '-o', 'id:10,mem:15'])
        return ResMEMCollector.parse_mem_per_lve(lveps_output.split('\n'))

    def execute(self, lve_data):
        self.update_lve_data(lve_data, self.get_mem_per_lve())

    @staticmethod
    def update_lve_data(lve_data, pmem_dict):
        """
        updates lve_data['stats'] pmem values for all LVEs
        Only update LVEs already in lve_data['stats'], skip others

        :param lve_data: plugin's lve_data
        :param pmem_dict: dictionary of LVE id <-> memory used in bytes
        :return: updated stats structure in lve_data
        """
        stats = lve_data['stats']
        for lve_id in stats:
            stat = stats[lve_id]
            if lve_id in pmem_dict:
                limit = stat.lmemphy
                pmem = pmem_dict[lve_id]
                if limit and pmem > limit:
                    stat.memphy = limit
                else:
                    stat.memphy = pmem_dict[lve_id]
            else:
                stat.memphy = 0
