#!/opt/cloudlinux/venv/bin/python3 -bb
# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#
"""
Domain POST modification hook.
This script must be called AFTER different domain-related
changes in control panel. See --help and docs for detailed
description of actions and parameters.
"""
import argparse

from clcommon.public_hooks.lib import ModifyDomainHook
from clcommon.public_hooks import setup_logger_and_sentry


if __name__ == '__main__':
    setup_logger_and_sentry()

    parser = argparse.ArgumentParser(description=__doc__)
    subparsers = parser.add_subparsers(dest='command')

    create_domain = subparsers.add_parser(
        'create', help="[NOT FOR VENDORS PANELS INTEGRATION] Call when domain record is created.")
    create_domain.add_argument('--username', '-u', required=True, help='User that owns domain.')
    create_domain.add_argument('--domain', '-ds', required=True,
                               help='Domain name e.g.: `userdomain.com`')

    modify_domain = subparsers.add_parser(
        'modify', help="Call when domain record is updated (e.g "
                       "all sites that have worked on old domain "
                       "now work on a new one).")
    modify_domain.add_argument('--username', '-u', required=True, help='User that owns domain.')
    modify_domain.add_argument('--domain', '-ds', required=True,
                               help='Domain name before rename, e.g.: `userdomain.com`')
    modify_domain.add_argument('--new-domain', '-dt', required=True,
                               help='Domain name after rename, e.g.: `userdomain2.eu`')
    # TODO: how to make this better?
    modify_domain.add_argument(
        '--include-subdomains', required=False, default=False, action='store_true',
        help='If set, we will also process all domains '
             'that end with domain name (subdomains). '
             'E.g. when renaming olddomain.com -> newdomain.eu '
             'we will also rename sub.olddomain.com to sub.newdomain.eu')

    modify_domain = subparsers.add_parser(
        'delete', help="[NOT FOR INTEGRATED CONTROL PANELS] Call when domain record is deleted.")
    modify_domain.add_argument('--username', '-u', required=True, help='User that owns domain.')
    modify_domain.add_argument('--domain', '-ds', required=True,
                               help='Domain name before rename, e.g.: `userdomain.com`')

    args = parser.parse_args()

    hook = ModifyDomainHook()
    if args.command == 'create':
        hook.post_create_domain(
            args.username, args.domain)
    elif args.command == 'modify':
        hook.post_modify_domain(
            args.username, args.domain, args.new_domain,
            include_subdomains=args.include_subdomains)
    elif args.command == 'delete':
        hook.post_delete_domain(args.username, args.domain)
    else:
        raise NotImplementedError
