import os
from typing import Optional, Tuple

from defence360agent.utils import check_run
from im360.subsys.panels.base import RemoteIPInterface

__all__ = []  # type: list

_REMOTEIP = "remoteip"
_RPAF = "rpaf"
_HTTPD_IS_NOT_RUNNING = "httpd is not running"
_NGINX_MNG_CMD = ("/usr/local/psa/admin/sbin/nginxmng", "-s")


async def _enable_httpd_module(name: str):
    await check_run(["plesk", "sbin", "httpd_modules_ctl", "-e", name])


async def _is_nginx_proxy_on() -> bool:
    return b"Enabled" in (
        await check_run(
            _NGINX_MNG_CMD,
            env=dict(os.environ, LC_ALL="C"),
        )
    )


class RemoteIP(RemoteIPInterface):
    _REMOTEIP_MODULE_NAMES: Tuple[bytes, ...] = (
        b"mod_remoteip",
        b"remoteip_module",
        b"rpaf_module",
    )

    async def remoteip_activated(self) -> bool:
        return await self._is_loaded_to_apache()

    async def remoteip_install(self) -> Optional[str]:
        for m in (_REMOTEIP, _RPAF):
            await _enable_httpd_module(m)
        if await _is_nginx_proxy_on():
            await check_run(["service", "nginx", "reload"])
        return None
