# -*- coding: utf-8 -*-
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#
import os
import logging
import subprocess
from pwd import getpwnam

from clcommon.cpapi import cpinfo, userdomains, get_main_username_by_uid, docroot
from clcommon.public_hooks import POST_MODIFY_USER, POST_MODIFY_DOMAIN, POST_MODIFY_PACKAGE

PLESK_CONFIG_PATH = "/etc/psa/psa.conf"
logger = logging.getLogger(__name__)


def domain_updated(old_domain_name, new_domain_name=None):
    """
    Triggered after any domain change
    """
    # If no new_domain_name was given
    # there is no need to run post_modify_domain hence early return 0
    if not new_domain_name or new_domain_name == old_domain_name:
        return 0

    # TODO: probably we can use cpapi here
    # username = domain_owner(new_domain_name)
    domain_path, _ = docroot(new_domain_name)
    uid = os.stat(domain_path).st_uid
    system_user = get_main_username_by_uid(uid)
    # Emulate getpwuid error exception
    if system_user == "N/A":
        raise KeyError(f"getpwuid(): uid not found: {uid}")

    args = [
        POST_MODIFY_DOMAIN,
        "modify",
        "--username",
        system_user,
        "--domain",
        old_domain_name,
        "--new-domain",
        new_domain_name,
    ]

    return subprocess.call(args)


def physical_hosting_created(new_system_user=None, new_domain_name=None):
    """
    Triggered after creation new subscription in Plesk.

    This function is called in two situations:
     - when user with main domain is created
     - and when additional domain created
    """
    if not new_system_user or not new_domain_name:
        logger.warning("ph_created but NEW_SYSTEM_USER or NEW_DOMAIN_NAME is empty, do nothing")
        return 0

    number_of_owned_domains = len(userdomains(new_system_user))
    # user must have at least one domain, because it is in his home dir path
    if number_of_owned_domains == 1:
        owner = cpinfo(cpuser=new_system_user, keyls=("reseller",))[0][0]
        return subprocess.call([POST_MODIFY_USER, "create", "--username", new_system_user, "--owner", owner])
    # in case when this method is called more than once -> additional domain created
    elif number_of_owned_domains > 1:
        return subprocess.call(
            [POST_MODIFY_DOMAIN, "create", "--username", new_system_user, "--domain", new_domain_name]
        )
    # impossible situation (I hope)
    else:
        raise ValueError(f"user {new_system_user} does not own any domains!")


def physical_hosting_deleted(system_user=None, domain_name=None):
    """
    Triggered after deletion physical hosting in Plesk.
    - means deletion of physical hosting on server
      (BUT NOT DOMAIN ITSELF -> domain can e.g. forward something)
    - when there are no more domains owned by system user -> unix
      user is destroyed here
    """
    if not system_user or not domain_name:
        logger.warning("ph_deleted but OLD_SYSTEM_USER or OLD_DOMAIN_NAME is empty, do nothing")
        return 0

    try:
        getpwnam(system_user)
    except KeyError:
        return subprocess.call([POST_MODIFY_USER, "delete", "--username", system_user])
    else:
        return subprocess.call([POST_MODIFY_DOMAIN, "delete", "--username", system_user, "--domain", domain_name])


def physical_hosting_updated(old_system_user=None, new_system_user=None):
    """
    Triggered after any subscription change in plesk.
    """
    if not old_system_user:
        logger.warning("ph_updated but OLD_SYSTEM_USER is empty, do nothing")
        return 0

    args = [POST_MODIFY_USER, "modify", "--username", old_system_user]
    if new_system_user != old_system_user:
        args += ["--new-username", new_system_user]

    return subprocess.call(args)


def plan_renamed():
    """
    Triggered after hosting plan is renamed.
    """
    package_id = os.environ.get("OLD_ADMIN_TEMPLATE")
    new_id = os.environ.get("NEW_ADMIN_TEMPLATE")
    if not package_id:
        package_id = os.environ.get("OLD_DOMAIN_TEMPLATE")
        new_id = os.environ.get("NEW_DOMAIN_TEMPLATE")

    # this id should be equal in case of modify\rename
    assert package_id and package_id == new_id

    args = [POST_MODIFY_PACKAGE, "rename", "--name", package_id]
    return subprocess.call(args)
