"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import functools
import logging
import inspect
import itertools
from pathlib import Path
from typing import Optional

logger = logging.getLogger(__name__)


def batched(iterable, n):
    # batched('ABCDEFG', 3) --> ABC DEF G
    if n < 1:
        raise ValueError("n must be at least one")
    it = iter(iterable)
    while batch := tuple(itertools.islice(it, n)):
        yield batch


def get_owner(path: Path) -> Optional[str]:
    try:
        return path.owner()
    except Exception:
        return None


def im360_present():
    try:
        import im360  # noqa: F401
    except ImportError:
        return False
    return True


def skip_for_im360(f):
    @functools.wraps(f)
    async def async_wrapper(*args, **kwargs):
        if im360_present():
            logger.info("Imunify360 is already installed. Skipping...")
            return
        return await f(*args, **kwargs)

    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        if im360_present():
            logger.info("Imunify360 is already installed. Skipping...")
            return
        return f(*args, **kwargs)

    return async_wrapper if inspect.iscoroutinefunction(f) else wrapper
