# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2023 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import email.utils
import logging
import os
import smtplib
import subprocess
from email.mime import base as mime_base, multipart
from logging.handlers import SysLogHandler
from typing import Optional

from _socket import error as socketerror

__author__ = "Aleksandr Shyshatsky"


class MailSendFailed(Exception):
    pass


class MailHelper:
    """Common class to send emails via local smtp or sendmail"""

    def __init__(self):
        if not logging.root.handlers:
            if os.path.exists('/dev/log'):
                SYS_LOG = SysLogHandler(address='/dev/log', facility=SysLogHandler.LOG_DAEMON)
                logging.root.addHandler(SYS_LOG)
                formatter_sys = logging.Formatter(fmt='mail_helper: (%(name)s) [%(levelname)s] %(message)s')
                SYS_LOG.setFormatter(formatter_sys)  # write in system log only ERROR-s and FATALLERROR-s
                SYS_LOG.setLevel(logging.ERROR)
            else:
                logging.basicConfig()
        self._log = logging.getLogger()

    def _sendmail_console(self, msg: mime_base.MIMEBase, encoding_name: Optional[str] = None):
        """
        Alternative function that should be called
        when we cannot send email via smtp
        :type msg: email.mime.base.MIMEBase
        :type encoding_name: encoding name to encode message text
        """
        try:
            if encoding_name:
                s_message = msg.as_string().encode(encoding_name)
            else:
                s_message = msg.as_string()
            cmd = ["/usr/sbin/sendmail", "-t", "-oi"]
            with subprocess.Popen(
                cmd,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True
            ) as proc:
                stdout = proc.communicate(s_message)[0]
                if proc.returncode != 0:
                    raise subprocess.CalledProcessError(proc.returncode, cmd, stdout)

        except (OSError, subprocess.CalledProcessError) as ex:
            raise MailSendFailed(str(ex)) from ex

    def _sendmail_smtp(self, from_addr, to_addrs, msg, encoding_name: Optional[str] = None):
        """
        Send email using smtp protocol;
        :type from_addr: str
        :type to_addrs: List[str]
        :type msg: email.mime.base.MIMEBase
        :type encoding_name: encoding name to encode message text
        """
        try:
            smtp_instance = smtplib.SMTP('localhost')
            if encoding_name:
                s_message = msg.as_string().encode(encoding_name)
            else:
                s_message = msg.as_string()
            smtp_instance.sendmail(from_addr, to_addrs, s_message)
            smtp_instance.quit()
        except (smtplib.SMTPException, socketerror) as ex:
            raise MailSendFailed(str(ex)) from ex

    def _add_message_id(self, from_addr: str, msg: multipart.MIMEMultipart):
        """Add Message-ID to message if it is missing
        :type from_addr: str
        :type msg: multipart.MIMEMultipart
        """
        if not msg['Message-ID']:
            msg['Message-ID'] = email.utils.make_msgid(domain=from_addr.split('@')[-1])
            self._log.debug('Added Message-ID: %s', msg['Message-ID'])

    def sendmail(self, from_addr, to_addrs, msg, encoding_name: Optional[str] = None):
        """
        Try to send message using local smtp;
        If that can't be done - try to use /usr/sbin/sendmail;
        :type from_addr: str
        :type to_addrs: List[str]
        :type msg: multipart.MIMEMultipart
        :type encoding_name: Optional[str] encoding name to encode message text
        """
        self._log.debug("Trying to send mail with params: %s, %s, %s", from_addr, to_addrs, msg)
        self._add_message_id(from_addr, msg)
        try:
            self._sendmail_smtp(from_addr, to_addrs, msg, encoding_name)
        except MailSendFailed as smtp_exception:
            self._log.debug("Smtp send failed, trying to send via sendmail")
            try:
                self._sendmail_console(msg)
            except MailSendFailed:
                self._log.exception("Cannot send email neither via smtp nor using sendmail; "
                                    "Please, check your mail server configuration")
                self._log.error(str(smtp_exception))
