# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import argparse
import lvestats
from datetime import datetime, timedelta
from lvestats.lib.commons.argparse_utils import check_timestamp, period_type2

DEFAULT_PERIOD_MINUTES = 10


def lve_read_snapshot_parser(now_=None):
    parser = argparse.ArgumentParser(
        prog='lve-read-snapshot',
        add_help=True,
        description='Reads lve system state snapshots for LVE/user',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    now = now_ or datetime.now()

    parser.add_argument(
        '--version', version=lvestats.__version__,
        help='Version number',
        dest='version',
        action='version')

    parser.add_argument(
        '-f',
        '--from',
        nargs='+',
        help='Run report from date and time in YYYY-MM-DD HH:MM format, if not present last 10 '
             'minutes are assumed',
        default=(now - timedelta(minutes=DEFAULT_PERIOD_MINUTES)).strftime('%Y-%m-%d %H:%M').split(),
        dest='ffrom', metavar='FROM')
    period_parser = parser.add_mutually_exclusive_group()
    period_parser.add_argument(
        '-t',
        '--to',
        nargs='+',
        help='Run report up to date and time in YYYY-MM-DD HH:MM format, if not present, '
             'reports results up to now',
        default=now.strftime('%Y-%m-%d %H:%M').split())
    period_parser.add_argument(
        '-p',
        '--period',
        type=lambda value: period_type2(value, now, False),
        help='Time period '
             'specify minutes with m,  h - hours, days with d, and values: '
             'today, yesterday, 5m - last 5 minutes, 4h - last four hours, '
             '2d - last 2 days, as well as today'
    )
    period_parser.add_argument(
        '--timestamp', type=check_timestamp,
        help='time stamp in unix format for get one snapshot')

    user_parser = parser.add_mutually_exclusive_group()  # use only -i or -u
    user_parser.add_argument(
        '-i',
        '--id',
        help='LVE id to show records for',
        required=False, type=int)
    user_parser.add_argument(
        '-u',
        '--user',
        help='user account to show records for',
        required=False)
    parser.add_argument(
        '-l',
        '--list',
        help='show timestamp list only',
        action='store_true',
        default=False)

    parser.add_argument(
        '-o',
        '--output',
        help='Filename to save snaphots report to, '
             'if not present, output will be sent to STDOUT',
        metavar='file')
    parser.add_argument(
        '-j',
        '--json',
        help='Output in json format',
        action='store_true',
        default=False)
    stats_parser = parser.add_argument_group()
    stats_parser.add_argument(
        '--stats',
        help='Output stats, instead of snapshots',
        action='store_true',
        default=False)
    stats_parser.add_argument(
        '--unit',
        help='Group stats by time unit. Example values 3h, 24h, 1d, 1w.'
             'Other possible value is "auto" for grouping by each incident.',
        metavar='unit',
        default='1d')
    return parser
