# -*- coding: utf-8 -*-

# license_lib.py

#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import subprocess
import os


class CloudlinuxLicenseLib:
    def __init__(self):
        self._CLDETECT_UTILITY = "/usr/bin/cldetect"
        self._CLNREG_KS_UTILITY = '/usr/sbin/clnreg_ks'
        self._CLNCHECK_UTILITY = '/usr/sbin/clncheck'

    def get_license_status(self):
        """
        Checks Cloudlinux license
        :return: True - license valid
                 False - invalid
        """
        # cldetect --check-license -q
        with subprocess.Popen(
            [self._CLDETECT_UTILITY, '--check-license', '-q'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        ) as proc:
            proc.communicate()
            return proc.returncode == 0

    def is_edition_migration_available(self):
        # check if edition migration is supported
        return os.path.isfile(self._CLNCHECK_UTILITY)

    def register_and_update_license_by_key(self, key):
        """
        Register and update Cloudlinux license by key
        :param key: License key
        :return:
        """
        # /usr/bin/cldetect --update-new-key key
        with subprocess.Popen(
            [self._CLDETECT_UTILITY, '--update-new-key', key],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        ) as proc:
            proc.communicate()

    def register_and_update_license_by_ip(self, stage=None):
        """
        Register and update Cloudlinux license by IP
        :param stage: Registraton stage.
            None - all stages. Normal mode
            'clnreg_ks' - only clnreg_ks. Only for tests
            'cldetect' - only cldetect. Only for tests
        :return:
        """
        # 1. /usr/sbin/clnreg_ks --force or /usr/sbin/clnreg_ks --force --migrate-silently
        if stage is None or stage == 'clnreg_ks':
            clnreg_cmd = [self._CLNREG_KS_UTILITY, '--force']
            if self.is_edition_migration_available():
                clnreg_cmd.append('--migrate-silently')
            with subprocess.Popen(
                clnreg_cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            ) as proc:
                proc.communicate()
        # 2. /usr/bin/cldetect --update-license
        if stage is None or stage == 'cldetect':
            with subprocess.Popen(
                [self._CLDETECT_UTILITY, '--update-license'],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            ) as proc:
                proc.communicate()
