# -*- coding: utf-8 -*-

import os
import configparser
import re

from typing import Dict  # NOQA

from clcommon.cpapi.cpapiexceptions import NotSupported
from clcommon.cpapi.plugins.universal import _dblogin_cplogin_pairs
from clcommon.features import Feature
from clcommon.cpapi.GeneralPanel import GeneralPanelPluginV1


__cpname__ = 'InterWorx'

PARSE_URI_PATTERN = re.compile(r'(.+)://(.+):(.+)@.*')


IWORX_INI = '/usr/local/interworx/iworx.ini'


# WARN: Probably will be deprecated for our "official" plugins.
# See pluginlib.detect_panel_fast()
def detect():
    return os.path.isfile(IWORX_INI)


def db_access(_conf_path=IWORX_INI):
    access = {}
    data_config = configparser.ConfigParser(strict=False)
    data_config.read(_conf_path)

    mysql_uri = data_config.get(section='mysql', option='rootdsn').strip('\"')
    parse_result = PARSE_URI_PATTERN.search(mysql_uri)
    (db, login, passwd) = (None, None, None)
    if parse_result:
        (db, login, passwd) = parse_result.group(1, 2, 3)
    access['login'] = login
    access['pass'] = passwd
    access['host'] = data_config.get(section='mysql', option='default_server').strip('\"')
    access['db'] = db
    return access


def cpusers():
    raise NotSupported({
        'message': '%(action)s is not currently supported.',
        'context': {'action': 'Getting all users registered in the Control Panel'}
    })


def dblogin_cplogin_pairs(cplogin_lst=None, with_system_users=False):
    access = db_access()
    data = _dblogin_cplogin_pairs(cplogin_lst=cplogin_lst, access=access)
    return data


def get_user_login_url(domain):
    return f'https://{domain}:2443'


class PanelPlugin(GeneralPanelPluginV1):
    def __init__(self):
        super().__init__()

    def getCPName(self):
        """
        Return panel name
        :return:
        """
        return __cpname__

    def get_cp_description(self):
        """
        Retrieve panel name and it's version
        :return: dict: { 'name': 'panel_name', 'version': 'panel_version', 'additional_info': 'add_info'}
            or None if can't get info
        """
        try:
            with open(IWORX_INI, "r", encoding='utf-8') as f:
                out = f.read()

            version = out.split('version')[4].replace('\n', '').replace('=', '').replace('"', '').split('[')[0]
            return {'name': __cpname__,
                    'version': version,
                    'additional_info': None}
        except Exception:
            return None

    def db_access(self):
        """
        Getting root access to mysql database.
        For example {'login': 'root', 'db': 'mysql', 'host': 'localhost', 'pass': '9pJUv38sAqqW'}

        :return: root access to mysql database
        :rtype: dict
        :raises: NoDBAccessData
        """
        return db_access()

    @staticmethod
    def useraliases(cpuser, domain):
        """
        Return aliases from user domain
        :param str|unicode cpuser: user login
        :param str|unicode domain:
        :return list of aliases
        """
        return []

    def cpusers(self):
        """
        Generates a list of cpusers registered in the control panel

        :return: list of cpusers registered in the control panel
        :rtype: tuple
        """
        return cpusers()

    def get_user_login_url(self, domain):
        """
        Get login url for current panel;
        :type domain: str
        :rtype: str
        """
        return get_user_login_url(domain)

    def get_unsupported_cl_features(self) -> tuple[Feature, ...]:
        return (
            Feature.RUBY_SELECTOR,
            Feature.PYTHON_SELECTOR,
            Feature.NODEJS_SELECTOR,
            Feature.RESELLER_LIMITS,
            Feature.WPOS,
        )

