#!/opt/cloudlinux/venv/bin/python3 -sbb

# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#

"""
This script is dedicated to save some cache data which we could
obtain fast, instead of iterating many users during our commands
Make sure script is called whatever status is changed to preserve
actual data
"""

import json
import argparse
import os
import logging

from secureio import write_file_via_tempfile
from clwpos.logsetup import setup_logging, init_wpos_sentry_safely
from clwpos.utils import is_wpos_supported, acquire_lock
from clwpos.constants import SUITES_CACHE

from clwpos.feature_suites.configurations import is_module_visible_for_user
from clwpos.optimization_features import ALL_OPTIMIZATION_FEATURES

_logger = setup_logging(
    caller_name='generate_accelerate_wp_cache',
    file_level=logging.INFO,
    logfile_path='/var/log/clwpos/generate_accelerate_wp_cache.log',
)

def generate_suites_statuses_cache():
    """
    Generates cache data for suites statuses
    """

    features_visible_for_any_user = [
        feature for feature in ALL_OPTIMIZATION_FEATURES
        if is_module_visible_for_user(feature)
    ]

    cache = {
        "suites_cache": {
            "ANY": {
                "visible": features_visible_for_any_user,
            }
        }
    }

    _logger.info('Saving cache with content=%s', str(cache))

    write_file_via_tempfile(json.dumps(cache), SUITES_CACHE, 0o600)

if __name__ == '__main__' and is_wpos_supported():
    with acquire_lock(os.path.join('/var/run/generate_accelerate_wp_cache.lock',), attempts=None):
        parser = argparse.ArgumentParser(description="Utility to collect information about user for AccelerateWP")
        parser.add_argument("--suites", default=False, action='store_true')
        args = parser.parse_args()

        init_wpos_sentry_safely()

        if args.suites:
            _logger.info("Generating suites statuses cache")
            try:
                generate_suites_statuses_cache()
            except Exception:
                import traceback
                traceback.print_exc()
                _logger.exception('Failed to generate suites statuses')

