from typing import List

from copy import deepcopy

from .exceptions import FeatureDisabledError
from .constants import CONFIG_MAPPINGS
from .model import FeatureManagementPerms


def check_feature(feature: str, permissions: List[str], user: str):
    """
    Raise exception if feature is disabled for user

    :param feature: feature name
    :param permissions: permissions required
    :param user: user name
    :return: None
    :raises FeatureDisabledError: if user is prohibited from use of the feature
    """
    if user is None:
        return

    perm = FeatureManagementPerms.get_perm(user)
    permission_value = perm.get_feature(feature)

    if permission_value not in permissions:
        raise FeatureDisabledError(
            "Feature '{name}' is disabled for user '{user}'".format(
                name=feature, user=user
            )
        )


def check_config(feature: str, user: str, section: str, raise_=False):
    """
    Check if section is allowed to be read or write by user

    :param feature: feature name
    :param user: user name
    :param section: section name to check
    :param raise_: True to raise exception, otherwise return True or False
    :return: True if config sections is allowed to user, False otherwise
    :raises FeatureDisabledError: raised if raise_=True
    """
    if user is None:
        return True

    if feature not in CONFIG_MAPPINGS:
        return True

    if section not in CONFIG_MAPPINGS[feature]:
        return True

    try:
        check_feature(feature, CONFIG_MAPPINGS[feature][section], user)
    except FeatureDisabledError:
        if raise_:
            raise
        return False

    return True


def config_cleanup(data: dict, user: str = None) -> dict:
    """
    Remove prohibited sections from user config

    :param data: config data
    :param user: user name
    :return: new config data
    """
    new_data = deepcopy(data)

    for feature in CONFIG_MAPPINGS:
        for section in data:
            if not check_config(feature, user, section):
                del new_data[section]

    return new_data


def config_validation(data: dict, user: str):
    """
    Raise exception if user is making changes on prohibited sections of config

    :param data: config data
    :param user: user name
    :return: None
    :raises FeatureDisabledError: if user is prohibited
    """
    for feature in CONFIG_MAPPINGS:
        for section in data:
            check_config(feature, user, section, raise_=True)
