"""
This module contains hook, which are called on feature management permission
changes. Note that hooks are not executed automatically, developer is
responsible to obtain specific hook using get_hook() function and call it.
To add hook, create function with name equal to feature name
"""
import functools
import logging
from typing import Any, Callable, Optional

from defence360agent.contracts.config import ConfigFile
from defence360agent.feature_management.constants import AV, FULL, PROACTIVE

logger = logging.getLogger(__name__)


def _hook_stub(*_):
    return True


def _result_warn(callback):
    @functools.wraps(callback)
    def wrap(user, value):
        result = callback(user, value)
        result or logger.warning(
            "Hook '%s(%s)' failed for user '%s'.",
            callback.__name__,
            value,
            user,
        )
        return result

    return wrap


@_result_warn
def antivirus(user: Optional[str], value: Any) -> bool:
    """Called when 'av' feature is changed"""
    if not user:
        return True

    config = ConfigFile()
    config_value = config.get("MALWARE_SCANNING", "default_action")

    user_config = ConfigFile(user)
    user_config_value = user_config.get("MALWARE_SCANNING", "default_action")

    if value == FULL:
        user_config_value = None
    elif (
        user_config_value
        and user_config_value.startswith("cleanup")
        and config_value
        and config_value.startswith("cleanup")
    ):
        user_config_value = "notify"

    try:
        user_config.set(
            "MALWARE_SCANNING", "default_action", user_config_value
        )
    except Exception:
        return False

    return True


@_result_warn
def proactive(user: Optional[str], value: Any) -> bool:
    """Called when 'proactive' feature is changed"""
    if not user:
        return True  # do nothing if no user specified

    config_value = None if value == FULL else "DISABLED"

    try:
        ConfigFile(user).set("PROACTIVE_DEFENCE", "mode", config_value)
    except Exception:
        return False

    return True


HOOKS = {
    AV: antivirus,
    PROACTIVE: proactive,
}


def get_hook(feature: str) -> Callable[[Optional[str], Any], bool]:
    """
    Get hook for specific feature. If no hook is implemented for this feature,
    return stub function
    :param feature: feature name
    :return: callable hook
    """
    return HOOKS.get(feature, _hook_stub)
