"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging
from pathlib import Path

from defence360agent.contracts.hook_events import HookEvent
from defence360agent.contracts.plugins import MessageSink, expect
from imav.malwarelib.config import MalwareScanType
from imav.malwarelib.utils.user_list import panel_users
from defence360agent.subsys import notifier

logger = logging.getLogger(__name__)


async def trigger_event(event_id: str, user: str, body: dict):
    try:
        await notifier.trigger_event(event_id, user, body)
    except Exception as e:
        logger.error("Failed to trigger an event '%s': %r", event_id, e)


async def trigger_scan_event(
    path: str,
    scan_id: str,
    scan_type: str,
    custom_event_id: str,
    user_event_id: str,
    **kwargs
):
    if scan_type == MalwareScanType.ON_DEMAND:
        user = ""
        event_id = custom_event_id
    elif scan_type in (MalwareScanType.BACKGROUND, MalwareScanType.USER):
        user = next(
            (
                u["user"]
                for u in await panel_users()
                if Path(path) == Path(u["home"])
            ),
            "",
        )
        event_id = user_event_id
    else:
        return

    await trigger_event(
        event_id,
        user,
        {"scan_id": scan_id, "path": path, **kwargs},
    )


class EventHooks(MessageSink):
    async def create_sink(self, loop):
        self._loop = loop

    @expect(HookEvent.MalwareDetectedCritical)
    async def malware_found(self, event):
        scan_type = event["scan_type"]

        if scan_type in (
            MalwareScanType.ON_DEMAND,
            MalwareScanType.RESCAN,
            MalwareScanType.RESCAN_OUTDATED,
        ):
            event_id = notifier.CUSTOM_SCAN_MALWARE_FOUND_EVENT_ID
        elif scan_type in (MalwareScanType.BACKGROUND, MalwareScanType.USER):
            event_id = notifier.USER_SCAN_MALWARE_FOUND_EVENT_ID
        else:
            return

        users = {hit["username"] for hit in event["DUMP"]}
        for user in users:
            await trigger_event(event_id, user, {"scan_id": event["scan_id"]})

    @expect(HookEvent.MalwareScanningStarted)
    async def scan_started(self, event):
        await trigger_scan_event(
            event["path"],
            event["scan_id"],
            event["scan_type"],
            notifier.CUSTOM_SCAN_STARTED_EVENT_ID,
            notifier.USER_SCAN_STARTED_EVENT_ID,
            started=event["started"],
        )

    @expect(HookEvent.MalwareScanningFinished)
    async def scan_finished(self, event):
        await trigger_scan_event(
            event["path"],
            event["scan_id"],
            event["scan_type"],
            notifier.CUSTOM_SCAN_FINISHED_EVENT_ID,
            notifier.USER_SCAN_FINISHED_EVENT_ID,
        )
