# -*- coding: utf-8 -*-
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#
import subprocess

import logging

from clcommon.public_hooks import (
    POST_MODIFY_USER,
    POST_MODIFY_DOMAIN,
    POST_MODIFY_ADMIN,
    PRE_MODIFY_USER
)

from clcommon.cpapi import cpinfo, is_admin
from clcommon.utils import is_user_present

ERR_NO_USER_FOUND = 'Unable to find system_user in hook env vars'

logger = logging.getLogger(__name__)


def domain_change_post(system_user, domain, new_domain):
    """
    Triggered on domain rename in DA
    """
    if system_user is None:
        logger.warning(ERR_NO_USER_FOUND)
        return 0
    return subprocess.call([
        POST_MODIFY_DOMAIN, 'modify',
        '--username', system_user,
        '--domain', domain,
        '--new-domain', new_domain, '--include-subdomains'])


def user_create_post(system_user):
    """
    Triggered after user/admin creation in directadmin
    """
    if system_user is None:
        logger.warning(ERR_NO_USER_FOUND)
        return 0

    exit_code = 0
    owner = cpinfo(cpuser=system_user, keyls=('reseller', ))[0][0]
    exit_code += subprocess.call([
        POST_MODIFY_USER, 'create',
        '--username', system_user, '--owner', owner])

    if is_admin(system_user):
        exit_code += subprocess.call([
            POST_MODIFY_ADMIN, 'create',
            '--name', system_user])
    return exit_code


def post_destroy_user(system_user, usertype):
    """
    Triggered after user modification in DA.
    """
    if system_user is None:
        logger.warning(ERR_NO_USER_FOUND)
        return 0

    exit_code = subprocess.call([
        POST_MODIFY_USER, 'delete', '--username', system_user])
    if usertype == 'admin':
        exit_code += subprocess.call([
            POST_MODIFY_ADMIN, 'delete',
            '--name', system_user])
    return exit_code


def user_destroy_pre(system_user, usertype):
    if system_user is None:
        logger.warning(ERR_NO_USER_FOUND)
        return 0

    if not is_user_present(system_user):
        logger.warning('User %s does not present in the system, skip hook', system_user)
        return 0

    return subprocess.call([
        PRE_MODIFY_USER, 'delete', '--username', system_user])

    # TODO: add if needed
    # if usertype == 'admin':
    #     subprocess.call([
    #         PRE_MODIFY_ADMIN, 'delete',
    #         '--name', system_user])


def user_restore_post(system_user):
    if system_user is None:
        logger.warning(ERR_NO_USER_FOUND)
        return 0

    return subprocess.call([
        POST_MODIFY_USER, 'restore',
        '--username', system_user])
