# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
import os
import re

from clcommon.utils import ExternalProgramFailed, run_command

MYSQL_BINARIES = [
    "/usr/sbin/mysqld",
    "/usr/libexec/mysqld",  # Fallback option
    "/usr/local/bin/mysqld",  # Present on CL9+DA
]


class MysqlInfo:
    def __init__(self):
        self.mysqld_v = self.retrieve_server_info()

    @staticmethod
    def get_binary():
        """
        Find mysql server binary
        :return: path to mysql binary
        """
        try:
            for binary in MYSQL_BINARIES:
                if os.path.exists(binary):
                    return binary
            return MYSQL_BINARIES[1]
        except OSError:
            return MYSQL_BINARIES[1]

    def retrieve_server_info(self):
        """
        Get server information through `<mysql_server_binary> -V` command
        :return: full command output
        """
        try:
            mysql_binary = self.get_binary()
            return run_command([mysql_binary, "-V"])
        except ExternalProgramFailed:
            # may be no such file or unknown error happen
            return None

    def get(self):
        """
        Return gathered data
        :return: dict(
            'vendor': MySQL|MariaDB|Percona,
            'version': server version in the form of {major}.{minor} or {major}.{minor}-{release} for percona,
            'cll-lve': patches from CL applied or not (True|False)
        )
        """
        return {"vendor": self.get_vendor(), "version": self.get_version(), "cll-lve": self.is_patched()}

    def get_vendor(self):
        """
        Extract MySQL vendor from server info
        :return: MySQL|MariaDB|Percona or
                `unknown` if failed to apply regex or
                None if there is no server info (this usually means that there are no MySQL installed)
        """
        if self.mysqld_v is None:
            return None
        # regex to find vendor -- something, starting from round bracket and followed by `Server`
        # (see test_clconfig_db_info_lib.py for detailed examples)
        p = re.compile(r"(?<=\().+(?=\sServer)")
        try:
            # we need only first word of first element
            return p.findall(self.mysqld_v)[0].split()[0]
        except IndexError:
            return "unknown"

    def get_version(self):
        """
        Retrieve MySQL server version from server info
        :return: X.X.X e.g. 10.2.16|5.6.39, includes release for percona (X.X.X-X.X e.g. 5.6.40-84.0) or
                `unknown` if failed to apply regex or
                None if there is no server info (this usually means that there are no MySQL installed)
        """
        if self.mysqld_v is None:
            return None
        # regex to find version -- numbers, divided by `.` or `-` which go after `Ver`
        # (see test_clconfig_db_info_lib.py for detailed examples)
        p = re.compile(r"(?<=Ver\s)[0-9\.\-]+")
        try:
            # we need first element, also should sanitize from trailing `-`
            return p.findall(self.mysqld_v)[0].rstrip("-")
        except IndexError:
            return "unknown"

    def is_patched(self):
        """
        Retrieve information about CL patch by `cll-lve` in server info
        :return: True if contains `cll-lve` seq
                False otherwise or
                None if there is no server info (this usually means that there are no MySQL installed)
        """
        if self.mysqld_v is None:
            return None
        return "cll-lve" in self.mysqld_v
