# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import os
import errno
from subprocess import Popen, PIPE
from clcommon.utils import ExternalProgramFailed

CAPTAIN_PATH = '/usr/share/python-cllib/cl.captain'


def captain(do_try, cmd, stdin=None):
    try:
        do_try()
    except (OSError, IOError) as e:
        if e.errno == errno.EDQUOT and os.geteuid() > 0:
            command = [CAPTAIN_PATH] + [str(s) for s in cmd]
            with Popen(
                command,
                stdin=None if stdin is None else PIPE,
                stdout=PIPE,
                stderr=PIPE,
                text=True,
            ) as proc:
                if stdin is None:
                    std_out, std_err = proc.communicate()
                else:
                    std_out, std_err = proc.communicate(stdin)

                if proc.returncode != 0:
                    raise ExternalProgramFailed(
                        std_err or f'output of "{command}": {std_out}'
                    ) from e
        else:
            raise


def mkdir(path, mode=0o755, recursive=False):
    umask = os.umask(0)
    if recursive:
        collector = "/"
        for subdir in path.split("/"):
            collector = os.path.join(collector, subdir)
            if os.path.exists(collector):
                continue
            captain(lambda: os.mkdir(collector, mode), ['-d', collector, '-m', f'{mode:o}'])
    else:
        captain(lambda: os.mkdir(path, mode), ['-d', path, '-m', f'{mode:o}'])
    os.umask(umask)


def write(path, content=''):
    def do_try():
        with open(path, 'w', encoding='utf-8') as fp:
            fp.write(content)
            fp.flush()
            os.fsync(fp.fileno())

    captain(do_try, ['-f', path], content)


def symlink(src, dst):
    captain(lambda: os.symlink(src, dst), ['-l', dst, '-s', src])
