# -*- coding: utf-8 -*-

# Command line arguments parser for cloudlinux-config utility

# cloudlinux-config Utility to check/set different Cloudlinux parameters
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from docopt import DocoptExit, docopt
from schema import And, Or, Schema, SchemaError

from clcommon.cpapi import is_reseller


def parse_cloudlinux_config_opts(argv, _is_json_need=False):
    # program name
    prog_name = "cloudlinux-config"

    docstring = f"""Utility to get/set any Cloudlinux config

    Usage:
        {prog_name} [get] [--json] [--for-reseller <str>]
        {prog_name} [set] [--json] --data <str> [--for-reseller <str>]
        {prog_name} [set] [--json] --reset-inodes-limit
        {prog_name} (-h | --help)

    Options:
        --json                  Return data in JSON format.
        --data <str>            Set parameters from JSON str
        --reset-inodes-limit    Reset inodes limits
        --for-reseller <str>    Set/Get configuration options for specific reseller (only lve-stats)
        -h, --help              Show this help message and exit
    """

    try:
        args = docopt(docstring, argv)
    except DocoptExit:
        s_error_string = 'ERROR: Invalid parameter passed'
        if not _is_json_need:
            s_error_string += "\n\n" + docstring
        return False, s_error_string
    if not args["get"] and not args["set"]:
        args["get"] = True

    # uncomment this for add full edit mode
    # gov_int_list = And(Use(lambda x:x.split(",")), lambda x:len(x) == 4,
    #                    Use(lambda x:map(int, x)))
    s = Schema({
        "set": bool,
        "get": bool,
        "--json": And(bool, lambda x: x, error="use --json option, other modes currently unsupported"),
        "--data": Or(None, str),
        "--for-reseller": And(Or(None, str), lambda x: x is None or is_reseller(x), error="non-existent reseller name"),
        "--reset-inodes-limit": bool,
        "--help": bool,
    })

    try:
        args = s.validate(args)
        status = True
    except SchemaError as e:
        args = str(e)
        status = False

    return status, args
