import json
import os
import re

from restore_infected.backup_backends.clusterlogics.exceptions import \
    TokenValidationError, TokenFileNotFoundError

TOKEN_FILE = '/var/restore_infected/clusterlogics_api_token.json'
CLUSTERLOGICS_TOKEN = '/usr/local/clusterlogics/etc/imunify360.json'
BAREOS_CONFIG = '/etc/bareos/bareos-fd.conf'


class TokenFile:
    KEYS_REQUIRED = ['username', 'apikey']
    KEYS = KEYS_REQUIRED + ['client', 'url']

    def __init__(self, token_file=TOKEN_FILE):
        self.file = token_file
        self._token = None

    @property
    def token(self):
        return self._token or self._validate(self._read())

    @token.setter
    def token(self, token):
        self._token = self._validate(token)
        self._save()

    @property
    def valid(self):
        try:
            self._validate(self._token)
        except (TokenFileNotFoundError, TokenValidationError):
            return False

        return True

    def _save(self):
        os.makedirs(os.path.dirname(self.file), exist_ok=True)
        with open(self.file, 'w') as token_file:
            json.dump(self.token, token_file)

    def _read(self):
        if not os.path.isfile(self.file):
            raise TokenFileNotFoundError(self.file)

        with open(self.file) as f:
            self._token = json.load(f)
            return self._token

    @classmethod
    def _validate(cls, token):
        if not token:
            raise TokenValidationError(msg='Invalid json content')

        if not token.get('client'):
            token['client'] = get_client()

        absent, empty = [], []
        for key in cls.KEYS:
            if key not in token:
                absent.append(key)
                token[key] = None
            elif key in cls.KEYS_REQUIRED and not token.get(key):
                empty.append(key)

        if empty or absent:
            raise TokenValidationError(empty, absent)

        return token


def check_existing_tokens():
    clusterlogics_token = TokenFile(CLUSTERLOGICS_TOKEN)
    backup_utils_token = TokenFile()
    if not backup_utils_token.valid and clusterlogics_token.valid:
        backup_utils_token.token = clusterlogics_token.token


def get_client():
    if os.path.isfile(BAREOS_CONFIG):
        with open(BAREOS_CONFIG) as f:
            bareos_config = f.read()
            start = bareos_config.find('FileDaemon')
            end = bareos_config.find('}', start)
            pattern = re.compile(r'Name\s*=\s*([.\w\d_-]*)\s')
            match = pattern.search(bareos_config, start, end)
            if match:
                name = match.group(1)
                return name
