import importlib

import time
from json import JSONDecodeError

import os
from urllib.parse import urljoin

import requests
from requests.auth import HTTPBasicAuth
from restore_infected.helpers import DateTime

from .exceptions import APIRequestError


def connection_warnings(enabled=False):
    if not enabled:
        importlib.import_module('requests.packages.urllib3').disable_warnings()


""" By default urllib3 bundled into requests prints a warning to console about
unverified https certificate when connecting to ClusterlogicsClient API so we
disable warnings during import """
connection_warnings(enabled=False)


class ClusterlogicsConnector:
    """ ClusterlogicsClient raw API wrapper """

    def __init__(self, username, apikey, client, url):
        self.username = username
        self.apikey = apikey
        self.client = client
        self.url = url

    @property
    def api_url(self):
        return urljoin(self.url, 'api/')

    def _build_url(self, *request):
        url = self.api_url
        for r in request:
            url = urljoin(url, r)
        return url.rstrip('/')

    def _api_request(self, *request, data=None):
        url = self._build_url(*request)
        auth = HTTPBasicAuth(self.username, self.apikey)
        res = requests.post(url, auth=auth, verify=False, data=data)
        try:
            r = res.json()
        except JSONDecodeError:
            res.raise_for_status()
        else:
            if isinstance(r, dict) and r.get('error'):
                raise APIRequestError(r.get('error'))
            return r

    def fileinfo(self, jobid, path):
        url = 'fileinfo/{}'.format(jobid)
        data = {'filename': path}
        return self._api_request(url, data=data)

    def jobs(self, until, path):
        url = 'jobs/{}'.format(self.client)
        if until:
            until = until.replace(microsecond=0)
        data = {
            'path': path,
            'from': until,
            'to': DateTime.fromtimestamp(int(time.time())),
        }
        return self._api_request(url, data=data)

    def restore(self, jobid, path, dst):
        url = 'restore/{}/{}'.format(jobid, self.client)
        data = {
            'files[]': [path],
            'stripprefix': os.path.dirname(path),
            'addprefix': dst,
            'home': ''
        }
        return self._api_request(url, data=data)

    def status(self, jobid):
        url = 'status/{}'.format(jobid)
        return self._api_request(url)

    def get_supported_systems(self):
        url = 'supportedsystems'
        return self._api_request(url)
