import importlib
import logging

from ..backup_backends_lib import asyncable, BackendNonApplicableError


logger = logging.getLogger(__name__)


def _stub(*_, **__):
    pass


def backend(name, async_=False):
    logger.debug('Importing backend %s (async=%r)', name, async_)

    rel_name = '.' + name

    try:
        backup_backend = importlib.import_module(rel_name, __name__)
    except (ImportError, ValueError):
        raise ValueError('unsupported backup backend: ' + name)

    if not backup_backend.is_suitable():
        raise BackendNonApplicableError(
            "backup backend can't be used: " + name)

    for func in ('init', 'cleanup', 'info', 'pre_backups'):
        if not hasattr(backup_backend, func):
            setattr(backup_backend, func, _stub)

    for func in ('init', 'backups', 'cleanup', 'info', 'pre_backups'):
        func_obj = getattr(backup_backend, func)
        if hasattr(func_obj, 'async_'):
            # `asyncable` already applied
            continue
        func_obj_async = asyncable(func_obj)
        setattr(backup_backend, func, func_obj_async)

    if async_:
        for entry in dir(backup_backend):
            entry_obj = getattr(backup_backend, entry)
            if hasattr(entry_obj, 'async_'):
                entry_obj.async_ = True

    return backup_backend
