"""This module implement inactivity tracker for ImunifyAV to automaticaly
shutdown the process when it is idle for certain time (no RPC calls and
long running tasks).
"""
import time
from contextlib import contextmanager, suppress
from logging import getLogger

logger = getLogger(__name__)


class InactivityTracker:
    def __init__(self):
        self._last_action_timestamp = time.monotonic()
        self._long_action_counter = 0
        self._long_actions_list = []
        self._timeout = 0

    def __str__(self):
        return "Time from last action is {:.0f}, long actions {}".format(
            time.monotonic() - self._last_action_timestamp,
            self._long_actions_list,
        )

    @contextmanager
    def task(self, name):
        self.start(name)
        try:
            yield
        finally:
            self.stop(name)

    def reset_timer(self):
        self._last_action_timestamp = time.monotonic()

    def start(self, name):
        self._long_action_counter += 1
        self._long_actions_list.append(name)
        self.reset_timer()

    def stop(self, name):
        self._long_action_counter -= 1
        with suppress(ValueError):
            self._long_actions_list.remove(name)
        self.reset_timer()

    def is_timeout(self):
        return (not self._long_action_counter) and (
            self._last_action_timestamp + self._timeout <= time.monotonic()
        )

    def set_timeout(self, timeout: int) -> None:
        self._timeout = timeout


track = InactivityTracker()
