# -*- coding: utf-8 -*-
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#
import os
import sys
from functools import partial
from logging.config import dictConfig

import logging
import yaml

from clsentry import init_cllib_sentry_client

BASE_DIR = os.path.abspath(os.path.dirname(__file__))
LOGGING_CONFIG = os.path.join(BASE_DIR, 'configs/logging.yaml')

CLOUDLINUX_HOOKS = '/usr/share/cloudlinux/hooks/'
POST_MODIFY_DOMAIN = os.path.join(CLOUDLINUX_HOOKS, 'post_modify_domain.py')
PRE_MODIFY_USER = os.path.join(CLOUDLINUX_HOOKS, 'pre_modify_user.py')
POST_MODIFY_USER = os.path.join(CLOUDLINUX_HOOKS, 'post_modify_user.py')
POST_MODIFY_ADMIN = os.path.join(CLOUDLINUX_HOOKS, 'post_modify_admin.py')
POST_MODIFY_PACKAGE = os.path.join(CLOUDLINUX_HOOKS, 'post_modify_package.py')


CONTACT_SUPPORT_MESSAGE_FOOTER = 'Please, contact CloudLinux support if you ' \
                                 'need help with resolving this issue.'

old_excepthook = None  # defined later


class NoTracebackFilter(logging.Filter):
    """Clear traceback from output"""

    def filter(self, record):
        record._exc_info_hidden, record.exc_info = record.exc_info, None
        # clear the exception traceback text cache, if created.
        record.exc_text = None
        return True


def _handle_unhandled_exception(logger, exc_type, exc_value, exc_traceback):
    """Handler for unhandled exceptions that will write to the logs"""
    if issubclass(exc_type, KeyboardInterrupt):
        # call the default excepthook saved at __excepthook__
        sys.__excepthook__(exc_type, exc_value, exc_traceback)
        return
    logger.critical("Unhandled exception", exc_info=(exc_type, exc_value, exc_traceback))
    return old_excepthook(exc_type, exc_value, exc_traceback)


def setup_logger_and_sentry():
    """
    Logger setup for all hook modules
    :rtype: logging.Logger
    """

    with open(LOGGING_CONFIG, 'r', encoding='utf-8') as f:
        config = yaml.safe_load(f.read())
        dictConfig(config)

    logger = logging.getLogger('clcommon.public_hooks')

    global old_excepthook
    old_excepthook = sys.excepthook
    sys.excepthook = partial(_handle_unhandled_exception, logger)

    # init sentry only after logging is ready
    # in order to hook it right
    # and disable it when flag is set (see commit msg)
    if not os.environ.get('DISABLE_SENTRY'):
        init_cllib_sentry_client()
    return logger
